﻿/* This client control encapsulate the functionalities of selecting Hengpi.
 * This control includes the logics of:
 * 1. Display Hengpi candidate list.
 * 2. Provide a input text box which allows customized user input Hengpi.
 * 3. Check the validity of hengpi against:
 *      a. Length violation
 *      b. Check if all chars in user provided Hengpi are chinese characters.
 */
 
// Register namespace
Type.registerNamespace("View");

// HengPiView Constructor
View.HengPiView = function(element) {

    // This HengPiView control should bind to a <div>
    // element in DOM tree.
    View.HengPiView.initializeBase(this, [element]);
    
    // Delegates for input HengPi textbox related event handler
    // Blur handler is invoked when user lose the focus from textbox
    this.m_dgBlurDelegate = null;
    // Focus handler is invoked when input textbox get focused.
    this.m_dgFocusDelegate = null;
    // Keyup handler for input textbox
    this.m_dgKeyupDelegate = null;
    
    // The valid length of Hengpi string
    this.m_iValidHengpiLength = 4;
    
    // Store a list of HengPi candidates generated by server
    this.m_rgHengpiCandidates = null;
    
    // DOM object which allow user input customized Hengpi
    this.m_txtinputhengpi = $get("inputHengpi");
    
    // Create the Hengpi candidate list control instance
    $create(View.CandidateList, {purpose: "Hengpi", groupSize: 10},  null ,null, $get("hp_cand"));
    this.m_candidatelist = $find("hp_cand");
    
}

// HengpiView prototype implementation
View.HengPiView.prototype = 
{
    // Properties
    set_validHengpiLength: function(iValidHengpiLength)
    {
        this.m_iValidHengpiLength = iValidHengpiLength;
    },
    
    get_validHengpiLength: function()
    {
        return this.m_iValidHengpiLength;
    },
    
    set_inputHengpi: function(inputHengpi)
    {
        this.m_txtinputhengpi.value = inputHengpi;
        this._changeHengpiInputBackground();
    },
    
    get_inputHengpi: function()
    {
        return this.m_txtinputhengpi.value;
    },
    
    set_result: function(rgResult)
    {
        this.m_rgHengpiCandidates = rgResult;
        this._constructHengPiView(rgResult);
    },
    
    get_inputHengpiElement: function()
    {
        return this.m_txtinputhengpi;
    },
    
    // Events for internal candidate list
    add_click: function(handler) 
    {
        this.m_candidatelist.add_click(handler);
    },
    
    remove_click: function(handler) 
    {
        this.m_candidatelist.remove_click(handler);
    },

    // Events for internal candidate list
    add_hover: function(handler) 
    {
        this.m_candidatelist.add_hover(handler);
    },
    
    remove_hover: function(handler) 
    {
        this.m_candidatelist.remove_hover(handler);
    },
    
    // Events for internal candidate list
    add_unhover: function(handler) 
    {
        this.m_candidatelist.add_unhover(handler);
    },
    
    remove_unhover: function(handler) 
    {
        this.m_candidatelist.remove_unhover(handler);
    },
    
    // Events for input Hengpi textbox
    add_keyup: function(handler) 
    {
        this.get_events().addHandler('keyup', handler);
    },
    
    remove_keyup: function(handler) 
    {
        this.get_events().removeHandler('keyup', handler);
    },
    
    // This initialze function will be called by the runtime when
    // this control is created. Here we create event delegete 
    // in initialize funtion. 
    initialize: function() 
    {
        if (this.m_dgBlurDelegate === null) {
            this.m_dgBlurDelegate = Function.createDelegate(this, this._blurHandler);
        }
        Sys.UI.DomEvent.addHandler(this.m_txtinputhengpi, 'blur', this.m_dgBlurDelegate);
        
        if (this.m_dgFocusDelegate === null) {
            this.m_dgFocusDelegate = Function.createDelegate(this, this._focusHandler);
        }
        Sys.UI.DomEvent.addHandler(this.m_txtinputhengpi, 'focus', this.m_dgFocusDelegate);
        
        if (this.m_dgKeyupDelegate === null) {
            this.m_dgKeyupDelegate = Function.createDelegate(this, this._keyupHandler);
        }
        Sys.UI.DomEvent.addHandler(this.m_txtinputhengpi, 'keyup', this.m_dgKeyupDelegate);

        View.HengPiView.callBaseMethod(this, 'initialize');

    },
    
    // Release resources before control is disposed. The main work here
    // is to remove the handler from DOM element and delete the 
    // event delegates created during intializtion phase.
    dispose: function() 
    {
        if (this.m_dgBlurDelegate)
        {
            Sys.UI.DomEvent.removeHandler(this.m_txtinputhengpi, 'blur', this.m_dgBlurDelegate);
            delete this.m_dgBlurDelegate;
        }
        
        if (this.m_dgFocusDelegate)
        {
            Sys.UI.DomEvent.removeHandler(this.m_txtinputhengpi, 'focus', this.m_dgFocusDelegate);
            delete this.m_dgFocusDelegate;
        }
        
        if (this.m_dgKeyupDelegate)
        {
            Sys.UI.DomEvent.removeHandler(this.m_txtinputhengpi, 'keyup', this.m_dgKeyupDelegate);
            delete this.m_dgKeyupDelegate;
        }
        
        View.HengPiView.callBaseMethod(this, 'dispose');
    },
    
    // These three internal defined handlers are used to 
    // create corresponding delegates. In this pattern, we
    // can decouple the control initialization logic from client
    // custom event handler assignment.
    _blurHandler: function(event) 
    {
        this._changeHengpiInputBackground();
    },
    
    _focusHandler: function(event)
    {
        this.m_txtinputhengpi.className = "hengpi";
    },
    
    _keyupHandler: function(event) 
    {
        var h = this.get_events().getHandler('keyup');
        var sender = this._getSender(event);
        if (h) h(sender, event);
    },
    
    /// <summary>
	/// This method is to check if the current Hengpi in the textbox
    /// comply to the validation rule.
	/// </summary>
    /// <accessibility>public</accessibility>
	/// <return value>True, if valid; otherwise, false.</return value>
    isCurrentHengpiValid: function()
    {
        var sCurrentHengpi = this.m_txtinputhengpi.value;
        
        if(sCurrentHengpi.length == 0)
        {
            return -1;
        }
        
        if(sCurrentHengpi.length != this.m_iValidHengpiLength)
        {
            return -2;
        }
        
        if(!this._isChineseString(sCurrentHengpi))
        {
            return -3;
        }
        return 0;

    },
    
    /// <summary>
	/// This method is to clear UI and internal state of Hnegpi control.
	/// </summary>
    /// <accessibility>public</accessibility>
    clear: function()
    {
        this.m_candidatelist.clear();
        this.m_rgHengpiCandidates=null;
    },
    
    /// <summary>
	/// This method is to unselect all the hengpi candidate in the
	/// candidate list view if user want to show couplet without Hengpi.
	/// </summary>
    /// <accessibility>public</accessibility>
    unselectAllCandidate: function()
    {
        if(this.m_rgHengpiCandidates == null)
        {
            return;
        }
        this.m_candidatelist.setSelectedCandidateId(-1);
    },
    
    /// <summary>
	/// This method is to set candidate id according to user's selection.
	/// </summary>
    /// <accessibility>public</accessibility>
	/// <param name="iSelectedCandidateId">iSelectedCandidateId represent which candidate is selected</param>
    setSelectedCandidateId: function(iSelectedCandidateId)
    {
        return this.m_candidatelist.setSelectedCandidateId(iSelectedCandidateId);
    },
    
    /// <summary>
	/// Given Hengpi result, display all the Hengpi candidates.
	/// </summary>
    /// <accessibility>private</accessibility>
	/// <param name="reResult">An array of Hengpi candidates</param>
    _constructHengPiView: function(rgResult) 
    {
        if (rgResult)
        {
            this.clear();
            
            this.m_rgHengpiCandidates = rgResult;
            
            for( i=0; i<rgResult.length; i++)
            {
                this.m_candidatelist.addCandidate(rgResult[i]);
            }
        }
    },
    
    /// <summary>
	/// This is to check whether all chars in string parameter are all
	/// Chinese characters. We ignore chinese SBC case ,that is 
	/// [\uFE30-\uFFA0]. [\u4E00-\u9FA5] means the normal chinese 
	/// character in unicode. Param "gi" means globle search 
	/// and ignore match case.
	/// </summary>
    /// <accessibility>private</accessibility>
	/// <param name="sHengpi">A Hengpi candidate string</param>
	/// <return value>If the string is a chinese, return true. otherwise false.</return value>
    _isChineseString: function(sHengpi)
    {
        var pattern = new RegExp("^([\u4E00-\u9FA5])*$","gi"); 
      
        return pattern.test(sHengpi);
    },

    /// <summary>
	/// This method is to determine what style should be attach
	/// to the input Hengpi textbox. This method is to implement
	/// the watermark effect.
	/// </summary>
    /// <accessibility>private</accessibility>
    _changeHengpiInputBackground: function()
    {
        if(this.m_txtinputhengpi.value.length == 0)
        {
            this.m_txtinputhengpi.className = "hengpi_original";     
        }
        else
        {
            this.m_txtinputhengpi.className = "hengpi";
        }
    },
    
    /// <summary>
	/// This method is to get the element who invoked current evnet.
	/// </summary>
    /// <accessibility>private</accessibility>
	/// <param name="evt">event object</param>
	/// <return value>If success, it will return DOM element</return value>
    _getSender: function(e) {
        
        var target = e.target;              // DOM standard event model
        if (!target)
        {
            target = e.srcElement;          // IE event model
        }
        
        return target;
    }
    
}
View.HengPiView.registerClass('View.HengPiView', Sys.UI.Control);

// Since this script is not loaded by System.Web.Handlers.ScriptResourceHandler
// invoke Sys.Application.notifyScriptLoaded to notify ScriptManager 
// that this is the end of the script.
if (typeof(Sys) !== 'undefined') Sys.Application.notifyScriptLoaded();
